/**
 * QuanChain Wallet - Background Service Worker
 * Implements DTQPE (Dynamic Transcendent Quantum Proof Encryption)
 *
 * Key Features:
 * - 64KB Parent Wallet (master entropy)
 * - Child key derivation for all 20 security levels
 * - HKDF-SHA256 key expansion
 * - Migration support between security levels
 */

// ============ Configuration ============
const RPC_ENDPOINTS = [
  'http://44.211.131.103:8545',
  'http://52.0.250.14:8545',
  'http://3.236.96.187:8545',
  'http://3.91.206.16:8545'
];

let currentEndpoint = RPC_ENDPOINTS[0];

// DTQPE Constants
const PARENT_WALLET_SIZE = 65536; // 64 KB
const PBKDF2_ITERATIONS = 600000;
const AUTO_LOCK_TIMEOUT = 15 * 60 * 1000; // 15 minutes

// Security Level Definitions (all 20 levels)
const SECURITY_LEVELS = {
  // Classical (Levels 1-5)
  1: { name: 'ECDSA-128', algorithm: 'ECDSA', quantumBits: 50, sigSize: 64, keySize: 32 },
  2: { name: 'ECDSA-192', algorithm: 'ECDSA', quantumBits: 75, sigSize: 96, keySize: 32 },
  3: { name: 'ECDSA-256', algorithm: 'Ed25519', quantumBits: 100, sigSize: 128, keySize: 32 },
  4: { name: 'ECDSA+RSA-2048', algorithm: 'ECDSA+RSA', quantumBits: 112, sigSize: 320, keySize: 64 },
  5: { name: 'ECDSA+RSA-4096', algorithm: 'ECDSA+RSA', quantumBits: 128, sigSize: 576, keySize: 64 },

  // Hybrid Post-Quantum (Levels 6-11)
  6: { name: 'Dilithium2+ECDSA', algorithm: 'Dilithium2+ECDSA', quantumBits: 150, sigSize: 2420, keySize: 96 },
  7: { name: 'Dilithium2+P256', algorithm: 'Dilithium2+ECDSA', quantumBits: 150, sigSize: 2484, keySize: 96 },
  8: { name: 'Dilithium3+P384', algorithm: 'Dilithium3+ECDSA', quantumBits: 175, sigSize: 3421, keySize: 128 },
  9: { name: 'Dilithium3+Ed25519', algorithm: 'Dilithium3+EdDSA', quantumBits: 175, sigSize: 3357, keySize: 128 },
  10: { name: 'Falcon-512+RSA', algorithm: 'Falcon512+RSA', quantumBits: 180, sigSize: 1202, keySize: 128 },
  11: { name: 'Falcon-512+P384', algorithm: 'Falcon512+ECDSA', quantumBits: 180, sigSize: 786, keySize: 128 },

  // Pure Post-Quantum (Levels 12-15)
  12: { name: 'SPHINCS+-128f', algorithm: 'SPHINCS+', quantumBits: 200, sigSize: 17088, keySize: 64 },
  13: { name: 'SPHINCS+-192f', algorithm: 'SPHINCS+', quantumBits: 225, sigSize: 35664, keySize: 96 },
  14: { name: 'SPHINCS+-256f', algorithm: 'SPHINCS+', quantumBits: 256, sigSize: 49856, keySize: 128 },
  15: { name: 'Dilithium5+Falcon-1024', algorithm: 'DualPQ', quantumBits: 300, sigSize: 5577, keySize: 256 },

  // Reserved for Future NIST Standards (Levels 16-20)
  16: { name: 'Reserved-NIST-L16', algorithm: 'Reserved', quantumBits: 350, sigSize: 32768, keySize: 512 },
  17: { name: 'Reserved-NIST-L17', algorithm: 'Reserved', quantumBits: 400, sigSize: 40960, keySize: 1024 },
  18: { name: 'Reserved-NIST-L18', algorithm: 'Reserved', quantumBits: 450, sigSize: 49152, keySize: 2048 },
  19: { name: 'Reserved-NIST-L19', algorithm: 'Reserved', quantumBits: 480, sigSize: 57344, keySize: 4096 },
  20: { name: 'Maximum-Security', algorithm: 'MaxSec', quantumBits: 512, sigSize: 65536, keySize: 8192 }
};

// Value ranges for automatic level recommendations (in USD)
const VALUE_THRESHOLDS = {
  1: 10,           // < $10
  3: 1000,         // < $1,000
  5: 10000,        // < $10,000
  7: 100000,       // < $100,000
  10: 500000,      // < $500,000
  12: 1000000,     // < $1M
  14: 10000000,    // < $10M
  15: 100000000,   // < $100M
  20: Infinity     // Unlimited
};

// Migration thresholds (safety factors)
const MIGRATION_AUTOMATIC_FACTOR = 1.5;
const MIGRATION_SUGGESTED_FACTOR = 3.0;

// LQCp/h (Logical Qubit Cost per Hour) - Economic quantum threat metric
// Current estimates based on hardware trajectory
const LQCPH_CURRENT = 5000000; // $5M per logical qubit-hour (2024 estimate)
const LQCPH_YEAR_DECLINE = 0.5; // 50% decline per year (Moore's law for quantum)

// Multi-wallet state (in-memory)
// Supports multiple parent wallets like MetaMask
let walletState = {
  // All parent wallets keyed by wallet ID
  wallets: {},              // { walletId: { parentEntropy, childWallets, name, createdAt } }
  activeWalletId: null,     // Currently selected wallet ID
  activeLevel: 5,           // Currently active security level within the active wallet
  isUnlocked: false,
  sessionPassword: null
};

// Get the active wallet's data
function getActiveWallet() {
  if (!walletState.activeWalletId || !walletState.wallets[walletState.activeWalletId]) {
    return null;
  }
  return walletState.wallets[walletState.activeWalletId];
}

// Get child wallets for the active wallet
function getActiveChildWallets() {
  const wallet = getActiveWallet();
  return wallet ? wallet.childWallets : {};
}

// Generate a unique wallet ID
function generateWalletId() {
  return 'wallet_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
}

let lockTimeout = null;

// ============ Message Handler ============
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  handleMessage(request)
    .then(response => sendResponse(response))
    .catch(error => sendResponse({ error: error.message }));
  return true;
});

async function handleMessage(request) {
  resetAutoLock();

  switch (request.type) {
    case 'GET_WALLET_STATUS':
      return await getWalletStatus();

    case 'CREATE_WALLET':
      return await createWallet(request.password);

    case 'IMPORT_WALLET':
      return await importWallet(request.entropy, request.password);

    case 'UNLOCK_WALLET':
      return await unlockWallet(request.password);

    case 'LOCK_WALLET':
      return lockWallet();

    case 'GET_ACCOUNTS':
      return getAccounts();

    case 'GET_CHILD_WALLET':
      return getChildWallet(request.level, request.index);

    case 'SET_ACTIVE_LEVEL':
      return setActiveLevel(request.level);

    case 'GET_ALL_LEVELS':
      return getAllLevels();

    case 'GET_BALANCE':
      return await getBalance(request.address);

    case 'GET_NETWORK_INFO':
      return await getNetworkInfo();

    case 'SWITCH_ENDPOINT':
      return switchEndpoint(request.endpoint);

    case 'EXPORT_PARENT_WALLET':
      return exportParentWallet(request.password);

    case 'SEND_TRANSACTION':
      return await sendTransaction(request.tx, request.password);

    case 'GET_MIGRATION_RECOMMENDATION':
      return getMigrationRecommendation(request.balance, request.currentLevel);

    case 'MIGRATE_FUNDS':
      return await migrateFunds(request.fromLevel, request.toLevel, request.amount, request.password);

    case 'GET_SECURITY_LEVEL_INFO':
      return getSecurityLevelInfo(request.level);

    case 'GET_TRANSACTIONS':
      return await getTransactions(request.address, request.limit, request.offset);

    // Multi-wallet management
    case 'GET_ALL_WALLETS':
      return getAllWallets();

    case 'CREATE_NEW_WALLET':
    case 'CREATE_ADDITIONAL_WALLET':
      return await createAdditionalWallet(request.name, request.password);

    case 'SWITCH_WALLET':
      return await switchWallet(request.walletId);

    case 'RENAME_WALLET':
      return await renameWallet(request.walletId, request.name);

    case 'DELETE_WALLET':
      return await deleteWallet(request.walletId, request.password);

    case 'IMPORT_ADDITIONAL_WALLET':
      return await importAdditionalWallet(request.entropy, request.name, request.password);

    default:
      throw new Error('Unknown message type: ' + request.type);
  }
}

// ============ Auto-Lock ============
function resetAutoLock() {
  if (lockTimeout) {
    clearTimeout(lockTimeout);
  }
  if (walletState.isUnlocked) {
    lockTimeout = setTimeout(() => {
      lockWallet();
      chrome.runtime.sendMessage({ type: 'WALLET_LOCKED' }).catch(() => {});
    }, AUTO_LOCK_TIMEOUT);
  }
}

// ============ DTQPE Core Functions ============

/**
 * Generate a new 64KB parent wallet with cryptographically secure entropy
 */
async function generateParentWallet() {
  // Generate 64KB of random entropy in chunks (browser limits single call)
  const entropy = new Uint8Array(PARENT_WALLET_SIZE);
  const chunkSize = 65536; // Max for single getRandomValues call

  for (let offset = 0; offset < PARENT_WALLET_SIZE; offset += chunkSize) {
    const remaining = Math.min(chunkSize, PARENT_WALLET_SIZE - offset);
    const chunk = crypto.getRandomValues(new Uint8Array(remaining));
    entropy.set(chunk, offset);
  }

  return entropy;
}

/**
 * Derive a child key for a specific security level using HKDF
 *
 * Formula: K_{c,l,i} = HKDF-SHA256("quanchain-dtqpe-level-{l}-index-{i}", parentEntropy[0:b_l])
 * Where b_l is proportional to the security level's key requirements
 */
async function deriveChildKey(parentEntropy, level, index) {
  const levelInfo = SECURITY_LEVELS[level];
  if (!levelInfo) {
    throw new Error(`Invalid security level: ${level}`);
  }

  // Calculate how much parent entropy to use based on security level
  // Higher levels use more of the parent entropy
  const entropyBytesToUse = Math.min(
    PARENT_WALLET_SIZE,
    Math.max(256, level * 3277) // ~3.2KB per level, min 256 bytes
  );

  const parentSlice = parentEntropy.slice(0, entropyBytesToUse);

  // Create context for HKDF
  const context = `quanchain-dtqpe-level-${level}-index-${index}`;
  const encoder = new TextEncoder();
  const info = encoder.encode(context);

  // Import parent entropy as key material
  const keyMaterial = await crypto.subtle.importKey(
    'raw',
    parentSlice,
    'HKDF',
    false,
    ['deriveBits', 'deriveKey']
  );

  // Derive the required key size for this security level
  const requiredKeySize = levelInfo.keySize;

  // Use HKDF to expand to required size
  const derivedBits = await crypto.subtle.deriveBits(
    {
      name: 'HKDF',
      hash: 'SHA-256',
      salt: encoder.encode(`QuanChain-DTQPE-v1-salt-${level}`),
      info: info
    },
    keyMaterial,
    requiredKeySize * 8 // bits
  );

  const keyBytes = new Uint8Array(derivedBits);

  // Generate address from derived key with proper checksum
  // Format: QC{level}_{payload}_{checksum}
  const addressHash = await crypto.subtle.digest('SHA-256', keyBytes);
  const payload = base58Encode(new Uint8Array(addressHash)); // Full 32-byte hash

  // Compute checksum: sha256(level || payload), then base58, take first 4 chars
  const checksumData = `${level}${payload}`;
  const checksumHash = await crypto.subtle.digest('SHA-256', encoder.encode(checksumData));
  const checksumBase58 = base58Encode(new Uint8Array(checksumHash));
  const checksum = checksumBase58.slice(0, 4);

  const address = `QC${level}_${payload}_${checksum}`;

  return {
    level,
    index,
    keyBytes,
    address,
    algorithm: levelInfo.algorithm,
    name: levelInfo.name,
    quantumBits: levelInfo.quantumBits,
    signatureSize: levelInfo.sigSize
  };
}

/**
 * Derive child wallets for all implemented security levels
 */
async function deriveAllChildWallets(parentEntropy, index = 0) {
  const children = {};

  // Derive for levels 1-15 (implemented)
  // Levels 16-20 are reserved but we derive them anyway for future use
  for (let level = 1; level <= 20; level++) {
    try {
      children[level] = await deriveChildKey(parentEntropy, level, index);
    } catch (e) {
      console.warn(`Failed to derive level ${level}:`, e);
    }
  }

  return children;
}

// ============ Wallet Functions ============

async function getWalletStatus() {
  const { hasWallet } = await chrome.storage.local.get('hasWallet');

  let activeAddress = null;
  let activeWalletName = null;
  const activeWallet = getActiveWallet();

  if (walletState.isUnlocked && activeWallet && activeWallet.childWallets[walletState.activeLevel]) {
    activeAddress = activeWallet.childWallets[walletState.activeLevel].address;
    activeWalletName = activeWallet.name;
  }

  return {
    hasWallet: !!hasWallet,
    isUnlocked: walletState.isUnlocked,
    address: activeAddress,
    activeLevel: walletState.activeLevel,
    activeWalletId: walletState.activeWalletId,
    activeWalletName: activeWalletName,
    totalWallets: Object.keys(walletState.wallets).length,
    supportedLevels: Object.keys(SECURITY_LEVELS).map(Number)
  };
}

async function createWallet(password) {
  if (!password || password.length < 8) {
    throw new Error('Password must be at least 8 characters');
  }

  // Generate 64KB parent wallet
  const parentEntropy = await generateParentWallet();

  // Derive all child wallets
  const childWallets = await deriveAllChildWallets(parentEntropy, 0);

  // Create first wallet ID
  const walletId = generateWalletId();

  // Set wallet state with multi-wallet structure
  walletState = {
    wallets: {
      [walletId]: {
        parentEntropy,
        childWallets,
        name: 'Wallet 1',
        createdAt: Date.now()
      }
    },
    activeWalletId: walletId,
    activeLevel: 5, // Default to level 5 (ECDSA+RSA-4096)
    isUnlocked: true,
    sessionPassword: password
  };

  // Save using multi-wallet storage format
  await saveAllWallets(password);

  resetAutoLock();

  // Return all derived addresses for all levels
  const addresses = {};
  for (const [level, wallet] of Object.entries(childWallets)) {
    addresses[level] = {
      address: wallet.address,
      algorithm: wallet.algorithm,
      name: wallet.name,
      quantumBits: wallet.quantumBits
    };
  }

  return {
    success: true,
    activeAddress: childWallets[5].address,
    activeLevel: 5,
    allAddresses: addresses,
    parentWalletSize: PARENT_WALLET_SIZE,
    walletId,
    walletName: 'Wallet 1'
  };
}

async function importWallet(entropyInput, password) {
  if (!password || password.length < 8) {
    throw new Error('Password must be at least 8 characters');
  }

  // Parse input - support both JSON structured format and raw hex
  let parentEntropy;
  let importedAddresses = null;

  try {
    // Try to parse as JSON first (structured format)
    let entropyHex;

    if (entropyInput.trim().startsWith('{')) {
      const parsed = JSON.parse(entropyInput);

      // Validate format
      if (parsed.format !== 'DTQPE-WALLET') {
        throw new Error('Invalid wallet format. Expected DTQPE-WALLET format.');
      }

      // Extract entropy from structured format
      entropyHex = parsed.wallet?.entropy || parsed.entropy;

      // Store addresses for verification
      importedAddresses = parsed.addresses;

      // Log import info
      console.log(`Importing DTQPE wallet v${parsed.version}`);
      console.log(`Protocol: ${parsed.protocol?.name} ${parsed.protocol?.version}`);
    } else {
      // Assume raw hex format
      entropyHex = entropyInput.trim();
    }

    // Convert hex string to Uint8Array
    parentEntropy = hexToBytes(entropyHex);

    if (parentEntropy.length !== PARENT_WALLET_SIZE) {
      throw new Error(`Invalid parent wallet size: expected ${PARENT_WALLET_SIZE} bytes, got ${parentEntropy.length}`);
    }
  } catch (e) {
    if (e.message.includes('Invalid wallet format') || e.message.includes('parent wallet size')) {
      throw e;
    }
    throw new Error('Invalid parent wallet data: ' + e.message);
  }

  // Derive all child wallets
  const childWallets = await deriveAllChildWallets(parentEntropy, 0);

  // Verify addresses match if imported from structured format
  if (importedAddresses) {
    let verified = 0;
    let mismatches = [];

    for (const [level, expectedAddress] of Object.entries(importedAddresses)) {
      const derivedWallet = childWallets[level];
      if (derivedWallet) {
        if (derivedWallet.address === expectedAddress) {
          verified++;
        } else {
          mismatches.push({
            level,
            expected: expectedAddress,
            derived: derivedWallet.address
          });
        }
      }
    }

    if (mismatches.length > 0) {
      console.warn('Address verification mismatches:', mismatches);
      throw new Error(
        `Address verification failed! ${mismatches.length} addresses don't match. ` +
        `This may indicate corrupted data or incompatible derivation. ` +
        `First mismatch at level ${mismatches[0].level}.`
      );
    }

    console.log(`Verified ${verified} addresses successfully`);
  }

  // Create wallet ID for imported wallet
  const walletId = generateWalletId();

  walletState = {
    wallets: {
      [walletId]: {
        parentEntropy,
        childWallets,
        name: 'Imported Wallet',
        createdAt: Date.now()
      }
    },
    activeWalletId: walletId,
    activeLevel: 5,
    isUnlocked: true,
    sessionPassword: password
  };

  // Save using multi-wallet storage format
  await saveAllWallets(password);

  resetAutoLock();

  return {
    success: true,
    activeAddress: childWallets[5].address,
    activeLevel: 5,
    verified: importedAddresses ? Object.keys(importedAddresses).length : 0,
    walletId,
    walletName: 'Imported Wallet'
  };
}

async function unlockWallet(password) {
  try {
    // Use loadAllWallets which handles both multi-wallet and legacy formats
    await loadAllWallets(password);

    resetAutoLock();

    const activeWallet = getActiveWallet();
    const activeChild = activeWallet?.childWallets[walletState.activeLevel];

    return {
      success: true,
      address: activeChild?.address || null,
      activeLevel: walletState.activeLevel,
      activeWalletId: walletState.activeWalletId,
      activeWalletName: activeWallet?.name || 'Wallet 1',
      totalWallets: Object.keys(walletState.wallets).length
    };
  } catch (e) {
    throw new Error('Invalid password');
  }
}

function lockWallet() {
  // Clear sensitive data from all wallets
  for (const wallet of Object.values(walletState.wallets)) {
    if (wallet.parentEntropy) {
      wallet.parentEntropy.fill(0);
    }
    for (const childWallet of Object.values(wallet.childWallets || {})) {
      if (childWallet.keyBytes) {
        childWallet.keyBytes.fill(0);
      }
    }
  }

  walletState = {
    wallets: {},
    activeWalletId: null,
    activeLevel: 5,
    isUnlocked: false,
    sessionPassword: null
  };

  if (lockTimeout) {
    clearTimeout(lockTimeout);
    lockTimeout = null;
  }

  return { success: true };
}

function getAccounts() {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  const activeWallet = getActiveWallet();
  if (!activeWallet) {
    throw new Error('No active wallet');
  }

  const activeChild = activeWallet.childWallets[walletState.activeLevel];

  return {
    accounts: [{
      name: `${activeWallet.name} (Level ${walletState.activeLevel})`,
      address: activeChild.address,
      level: walletState.activeLevel,
      algorithm: activeChild.algorithm
    }],
    activeAccount: 0,
    activeLevel: walletState.activeLevel,
    activeWalletId: walletState.activeWalletId,
    activeWalletName: activeWallet.name
  };
}

function getChildWallet(level, index = 0) {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  const activeWallet = getActiveWallet();
  if (!activeWallet) {
    throw new Error('No active wallet');
  }

  const childWallet = activeWallet.childWallets[level];
  if (!childWallet) {
    throw new Error(`No wallet derived for level ${level}`);
  }

  return {
    level: childWallet.level,
    address: childWallet.address,
    algorithm: childWallet.algorithm,
    name: childWallet.name,
    quantumBits: childWallet.quantumBits,
    signatureSize: childWallet.signatureSize
  };
}

async function setActiveLevel(level) {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  if (!SECURITY_LEVELS[level]) {
    throw new Error(`Invalid security level: ${level}`);
  }

  const activeWallet = getActiveWallet();
  if (!activeWallet) {
    throw new Error('No active wallet');
  }

  walletState.activeLevel = level;

  await chrome.storage.local.set({ activeLevel: level });

  const childWallet = activeWallet.childWallets[level];

  return {
    success: true,
    activeLevel: level,
    address: childWallet.address,
    algorithm: childWallet.algorithm
  };
}

function getAllLevels() {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  const activeWallet = getActiveWallet();
  if (!activeWallet) {
    throw new Error('No active wallet');
  }

  const levels = {};
  for (const [level, childWallet] of Object.entries(activeWallet.childWallets)) {
    levels[level] = {
      address: childWallet.address,
      algorithm: childWallet.algorithm,
      name: childWallet.name,
      quantumBits: childWallet.quantumBits,
      signatureSize: childWallet.signatureSize,
      isImplemented: parseInt(level) <= 15,
      isActive: parseInt(level) === walletState.activeLevel
    };
  }

  return {
    levels,
    activeLevel: walletState.activeLevel,
    implementedLevels: 15,
    totalLevels: 20
  };
}

function getSecurityLevelInfo(level) {
  const info = SECURITY_LEVELS[level];
  if (!info) {
    throw new Error(`Invalid security level: ${level}`);
  }

  return {
    level,
    ...info,
    isImplemented: level <= 15,
    isClassical: level <= 5,
    isHybrid: level >= 6 && level <= 11,
    isPostQuantum: level >= 12 && level <= 15,
    isReserved: level >= 16
  };
}

async function getBalance(address) {
  const activeWallet = getActiveWallet();
  const addr = address || (walletState.isUnlocked && activeWallet?.childWallets[walletState.activeLevel]
    ? activeWallet.childWallets[walletState.activeLevel].address
    : null);

  if (!addr) {
    return { address: null, balance: '0' };
  }

  try {
    const result = await rpcCall('chain_getBalance', [addr]);
    return {
      address: addr,
      balance: result.balance || '0',
      level: walletState.activeLevel
    };
  } catch (e) {
    return { address: addr, balance: '0', error: e.message };
  }
}

async function getNetworkInfo() {
  try {
    const result = await rpcCall('net_nodeInfo', []);
    return {
      connected: true,
      endpoint: currentEndpoint,
      nodeInfo: result
    };
  } catch (e) {
    return {
      connected: false,
      endpoint: currentEndpoint,
      error: e.message
    };
  }
}

async function getTransactions(address, limit = 10, offset = 0) {
  const activeWallet = getActiveWallet();
  const addr = address || (walletState.isUnlocked && activeWallet?.childWallets[walletState.activeLevel]
    ? activeWallet.childWallets[walletState.activeLevel].address
    : null);

  if (!addr) {
    return { address: null, transactions: [], total: 0 };
  }

  try {
    const result = await rpcCall('chain_getTransactionsByAddress', [addr, limit, offset]);

    // Format transactions for display
    const transactions = (result?.transactions || []).map(tx => ({
      hash: tx.hash || '',
      from: tx.from || '',
      to: tx.to || '',
      value: tx.value || '0',
      blockHeight: tx.block_height || 0,
      channel: tx.channel || 1,
      timestamp: tx.timestamp || null,
      direction: tx.from === addr ? 'out' : 'in',
      status: 'confirmed'
    }));

    return {
      address: addr,
      transactions,
      total: result?.total || transactions.length,
      hasMore: result?.has_more || false
    };
  } catch (e) {
    console.error('Failed to get transactions:', e);
    return { address: addr, transactions: [], total: 0, error: e.message };
  }
}

function switchEndpoint(endpoint) {
  if (RPC_ENDPOINTS.includes(endpoint)) {
    currentEndpoint = endpoint;
    return { success: true, endpoint };
  }
  throw new Error('Invalid endpoint');
}

function exportParentWallet(password) {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  if (password !== walletState.sessionPassword) {
    throw new Error('Invalid password');
  }

  const activeWallet = getActiveWallet();
  if (!activeWallet) {
    throw new Error('No active wallet');
  }

  // Return the parent entropy with full derivation metadata for cross-wallet compatibility
  const entropyHex = bytesToHex(activeWallet.parentEntropy);

  // Build the standardized DTQPE wallet export format
  const walletExport = {
    // Format identification
    format: 'DTQPE-WALLET',
    version: 1,

    // Protocol specification
    protocol: {
      name: 'QuanChain DTQPE',
      version: '1.0.0',
      specification: 'Dynamic Transcendent Quantum Proof Encryption'
    },

    // Derivation parameters (critical for reconstruction)
    derivation: {
      function: 'HKDF-SHA256',
      saltPrefix: 'QuanChain-DTQPE-v1-salt-',
      infoPrefix: 'quanchain-dtqpe-level-',
      infoFormat: 'quanchain-dtqpe-level-{level}-index-{index}',
      entropyBytesFormula: 'min(65536, max(256, level * 3277))',
      addressFormat: 'QC{level}_{base58(SHA256(key)[0:20])}'
    },

    // Security level definitions
    levels: {
      total: 20,
      implemented: 15,
      reserved: [16, 17, 18, 19, 20],
      keySizes: {
        1: 32, 2: 32, 3: 32, 4: 64, 5: 64,
        6: 96, 7: 96, 8: 128, 9: 128, 10: 128, 11: 128,
        12: 64, 13: 96, 14: 128, 15: 256,
        16: 512, 17: 1024, 18: 2048, 19: 4096, 20: 8192
      },
      algorithms: {
        '1-5': 'Classical (ECDSA/RSA)',
        '6-11': 'Hybrid (Dilithium/Falcon + ECDSA)',
        '12-15': 'Post-Quantum (SPHINCS+/DualPQ)',
        '16-20': 'Reserved (Future NIST standards)'
      }
    },

    // Parent wallet data
    wallet: {
      entropy: entropyHex,
      size: PARENT_WALLET_SIZE,
      sizeHex: entropyHex.length,
      checksum: null, // Will be computed below
      createdAt: activeWallet.createdAt,
      exportedAt: Date.now()
    },

    // Derived addresses (for verification)
    addresses: {}
  };

  // Compute checksum (first 8 chars of SHA256 of entropy)
  crypto.subtle.digest('SHA-256', activeWallet.parentEntropy).then(hash => {
    walletExport.wallet.checksum = bytesToHex(new Uint8Array(hash).slice(0, 4));
  });

  // Include all derived addresses for verification
  for (const [level, wallet] of Object.entries(activeWallet.childWallets)) {
    walletExport.addresses[level] = wallet.address;
  }

  return {
    // Full structured export (JSON)
    exportData: walletExport,

    // Also return raw entropy for simple import
    entropy: entropyHex,

    // Human-readable export string
    exportString: JSON.stringify(walletExport, null, 2),

    warning: 'CRITICAL: This export contains your 64KB parent wallet and complete derivation parameters. Anyone with this data can derive ALL your addresses at ALL 20 security levels. Store securely offline!'
  };
}

// ============ Multi-Wallet Management Functions ============

/**
 * Get all wallets (for wallet selector dropdown)
 * Returns wallet summaries without sensitive data
 */
function getAllWallets() {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  const wallets = [];
  for (const [walletId, wallet] of Object.entries(walletState.wallets)) {
    // Get the default address (level 5) for display
    const defaultAddress = wallet.childWallets[5]?.address || wallet.childWallets[1]?.address || '';

    wallets.push({
      id: walletId,
      name: wallet.name || `Wallet ${wallets.length + 1}`,
      address: defaultAddress,
      createdAt: wallet.createdAt,
      isActive: walletId === walletState.activeWalletId
    });
  }

  return {
    wallets,
    activeWalletId: walletState.activeWalletId,
    totalWallets: wallets.length
  };
}

/**
 * Create a new parent wallet
 * Generates new 64KB entropy and derives all child wallets
 */
async function createNewWallet(name, password) {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  if (password !== walletState.sessionPassword) {
    throw new Error('Invalid password');
  }

  // Generate new 64KB parent wallet
  const parentEntropy = await generateParentWallet();

  // Derive all child wallets
  const childWallets = await deriveAllChildWallets(parentEntropy, 0);

  // Create wallet ID
  const walletId = generateWalletId();
  const walletName = name || `Wallet ${Object.keys(walletState.wallets).length + 1}`;

  // Add to wallets
  walletState.wallets[walletId] = {
    parentEntropy,
    childWallets,
    name: walletName,
    createdAt: Date.now()
  };

  // Switch to the new wallet
  walletState.activeWalletId = walletId;

  // Persist to storage
  await saveAllWallets(password);

  return {
    success: true,
    walletId,
    name: walletName,
    address: childWallets[5].address,
    totalWallets: Object.keys(walletState.wallets).length
  };
}

/**
 * Switch to a different wallet
 */
async function switchWallet(walletId) {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  if (!walletState.wallets[walletId]) {
    throw new Error('Wallet not found');
  }

  walletState.activeWalletId = walletId;

  // Save active wallet preference
  await chrome.storage.local.set({ activeWalletId: walletId });

  const wallet = walletState.wallets[walletId];
  const activeChild = wallet.childWallets[walletState.activeLevel];

  return {
    success: true,
    walletId,
    name: wallet.name,
    address: activeChild.address,
    activeLevel: walletState.activeLevel
  };
}

/**
 * Rename a wallet
 */
async function renameWallet(walletId, name) {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  if (!walletState.wallets[walletId]) {
    throw new Error('Wallet not found');
  }

  if (!name || name.trim().length === 0) {
    throw new Error('Name cannot be empty');
  }

  walletState.wallets[walletId].name = name.trim();

  // Persist to storage
  await saveAllWallets(walletState.sessionPassword);

  return {
    success: true,
    walletId,
    name: walletState.wallets[walletId].name
  };
}

/**
 * Delete a wallet
 * Cannot delete the last wallet
 */
async function deleteWallet(walletId, password) {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  if (password !== walletState.sessionPassword) {
    throw new Error('Invalid password');
  }

  if (!walletState.wallets[walletId]) {
    throw new Error('Wallet not found');
  }

  const walletCount = Object.keys(walletState.wallets).length;
  if (walletCount <= 1) {
    throw new Error('Cannot delete the last wallet');
  }

  // Clear sensitive data before deleting
  const wallet = walletState.wallets[walletId];
  if (wallet.parentEntropy) {
    wallet.parentEntropy.fill(0);
  }
  for (const childWallet of Object.values(wallet.childWallets)) {
    if (childWallet.keyBytes) {
      childWallet.keyBytes.fill(0);
    }
  }

  // Delete the wallet
  delete walletState.wallets[walletId];

  // If we deleted the active wallet, switch to another
  if (walletState.activeWalletId === walletId) {
    walletState.activeWalletId = Object.keys(walletState.wallets)[0];
  }

  // Persist to storage
  await saveAllWallets(password);

  return {
    success: true,
    deletedWalletId: walletId,
    activeWalletId: walletState.activeWalletId,
    remainingWallets: Object.keys(walletState.wallets).length
  };
}

/**
 * Import an additional wallet from entropy
 */
async function importAdditionalWallet(entropyInput, name, password) {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  if (password !== walletState.sessionPassword) {
    throw new Error('Invalid password');
  }

  // Parse the entropy input (same logic as importWallet)
  let parentEntropy;
  let importedAddresses = null;

  try {
    let entropyHex;

    if (entropyInput.trim().startsWith('{')) {
      const parsed = JSON.parse(entropyInput);

      if (parsed.format !== 'DTQPE-WALLET') {
        throw new Error('Invalid wallet format. Expected DTQPE-WALLET format.');
      }

      entropyHex = parsed.wallet?.entropy || parsed.entropy;
      importedAddresses = parsed.addresses;
    } else {
      entropyHex = entropyInput.trim();
    }

    parentEntropy = hexToBytes(entropyHex);

    if (parentEntropy.length !== PARENT_WALLET_SIZE) {
      throw new Error(`Invalid parent wallet size: expected ${PARENT_WALLET_SIZE} bytes, got ${parentEntropy.length}`);
    }
  } catch (e) {
    if (e.message.includes('Invalid wallet format') || e.message.includes('parent wallet size')) {
      throw e;
    }
    throw new Error('Invalid parent wallet data: ' + e.message);
  }

  // Derive all child wallets
  const childWallets = await deriveAllChildWallets(parentEntropy, 0);

  // Verify addresses if available
  if (importedAddresses) {
    for (const [level, expectedAddress] of Object.entries(importedAddresses)) {
      const derivedWallet = childWallets[level];
      if (derivedWallet && derivedWallet.address !== expectedAddress) {
        throw new Error(`Address verification failed at level ${level}`);
      }
    }
  }

  // Check for duplicate wallets (by comparing level 5 addresses)
  const newAddress = childWallets[5]?.address;
  for (const existingWallet of Object.values(walletState.wallets)) {
    if (existingWallet.childWallets[5]?.address === newAddress) {
      throw new Error('This wallet has already been imported');
    }
  }

  // Create wallet ID and add to wallets
  const walletId = generateWalletId();
  const walletName = name || `Imported Wallet ${Object.keys(walletState.wallets).length + 1}`;

  walletState.wallets[walletId] = {
    parentEntropy,
    childWallets,
    name: walletName,
    createdAt: Date.now()
  };

  // Switch to the new wallet
  walletState.activeWalletId = walletId;

  // Persist to storage
  await saveAllWallets(password);

  return {
    success: true,
    walletId,
    name: walletName,
    address: childWallets[5].address,
    verified: importedAddresses ? Object.keys(importedAddresses).length : 0,
    totalWallets: Object.keys(walletState.wallets).length
  };
}

/**
 * Save all wallets to encrypted storage
 */
async function saveAllWallets(password) {
  const walletsToStore = {};
  const walletNames = {};

  for (const [walletId, wallet] of Object.entries(walletState.wallets)) {
    // Encrypt each wallet's parent entropy separately
    const encrypted = await encryptParentWallet(wallet.parentEntropy, password);
    walletsToStore[walletId] = {
      encrypted,
      createdAt: wallet.createdAt
    };
    walletNames[walletId] = wallet.name;
  }

  await chrome.storage.local.set({
    hasWallet: true,
    multiWallet: true,
    encryptedWallets: walletsToStore,
    walletNames,
    activeWalletId: walletState.activeWalletId,
    activeLevel: walletState.activeLevel
  });
}

/**
 * Load all wallets from encrypted storage
 */
async function loadAllWallets(password) {
  const data = await chrome.storage.local.get([
    'multiWallet', 'encryptedWallets', 'walletNames',
    'activeWalletId', 'activeLevel', 'encryptedWallet'
  ]);

  // Check if this is the new multi-wallet format
  if (data.multiWallet && data.encryptedWallets) {
    const wallets = {};

    for (const [walletId, walletData] of Object.entries(data.encryptedWallets)) {
      const parentEntropy = await decryptParentWallet(walletData.encrypted, password);
      const childWallets = await deriveAllChildWallets(parentEntropy, 0);

      wallets[walletId] = {
        parentEntropy,
        childWallets,
        name: data.walletNames?.[walletId] || `Wallet`,
        createdAt: walletData.createdAt
      };
    }

    walletState = {
      wallets,
      activeWalletId: data.activeWalletId || Object.keys(wallets)[0],
      activeLevel: data.activeLevel || 5,
      isUnlocked: true,
      sessionPassword: password
    };
  } else if (data.encryptedWallet) {
    // Legacy single-wallet format - migrate to multi-wallet
    const parentEntropy = await decryptParentWallet(data.encryptedWallet, password);
    const childWallets = await deriveAllChildWallets(parentEntropy, 0);

    const walletId = generateWalletId();

    walletState = {
      wallets: {
        [walletId]: {
          parentEntropy,
          childWallets,
          name: 'Wallet 1',
          createdAt: data.encryptedWallet.createdAt || Date.now()
        }
      },
      activeWalletId: walletId,
      activeLevel: data.activeLevel || 5,
      isUnlocked: true,
      sessionPassword: password
    };

    // Migrate to new format
    await saveAllWallets(password);
  } else {
    throw new Error('No wallet found');
  }

  return walletState;
}

// ============ Migration Functions ============

/**
 * Calculate LQCp/h (Logical Qubit Cost per Hour) based cracking cost
 *
 * Formula: CrackingCost = LQCp/h × Q_l × T_l
 * Where:
 *   - Q_l = qubits needed for level l (based on quantum bits)
 *   - T_l = time needed to crack level l
 *
 * Safety Margin = CrackingCost / Balance
 *   - < 1.5x: Automatic migration required
 *   - < 3.0x: Migration suggested
 */
function calculateLQCphCrackingCost(level, yearsFromNow = 0) {
  const levelInfo = SECURITY_LEVELS[level];

  // Project LQCp/h into the future (declines over time)
  const futureLqcph = LQCPH_CURRENT * Math.pow(LQCPH_YEAR_DECLINE, yearsFromNow);

  // Calculate qubits needed based on quantum bit security
  // Classical levels (1-5): ECDSA can be broken with ~160-256 qubits using Shor's algorithm
  // Hybrid levels (6-11): Require both classical and quantum attacks
  // PQ levels (12-15): Require Grover's algorithm (quadratic speedup only)
  let qubitsNeeded;
  let hoursNeeded;

  if (level <= 5) {
    // Classical ECDSA/RSA - vulnerable to Shor's algorithm
    // ~2n qubits for n-bit security
    qubitsNeeded = levelInfo.quantumBits * 2;
    hoursNeeded = 1; // Shor is fast once you have the qubits
  } else if (level <= 11) {
    // Hybrid - both classical and PQ components need to be broken
    // Classical part: Shor's algorithm
    // PQ part: Grover's algorithm (quadratic speedup)
    const classicalQubits = 256; // For the classical component
    const pqQubits = Math.pow(2, levelInfo.quantumBits / 128); // Grover speedup
    qubitsNeeded = Math.max(classicalQubits, pqQubits);
    hoursNeeded = Math.pow(2, levelInfo.quantumBits / 50); // Much longer for hybrid
  } else {
    // Pure post-quantum - only Grover attack applies
    // Grover provides quadratic speedup: 2^n -> 2^(n/2) operations
    qubitsNeeded = Math.pow(2, levelInfo.quantumBits / 256);
    hoursNeeded = Math.pow(2, levelInfo.quantumBits / 25); // Very long time
  }

  return futureLqcph * qubitsNeeded * hoursNeeded;
}

function getMigrationRecommendation(balanceUsd, currentLevel) {
  const levelInfo = SECURITY_LEVELS[currentLevel];

  // Calculate current cracking cost
  const currentCrackingCost = calculateLQCphCrackingCost(currentLevel, 0);

  // Calculate cracking cost 5 years from now (quantum progress)
  const futureCrackingCost = calculateLQCphCrackingCost(currentLevel, 5);

  // Safety margin based on current quantum capabilities
  const currentSafetyMargin = currentCrackingCost / Math.max(balanceUsd, 1);

  // Safety margin based on projected quantum capabilities
  const futureSafetyMargin = futureCrackingCost / Math.max(balanceUsd, 1);

  let urgency = 'none';
  let deadline = null;
  let message = '';

  if (currentSafetyMargin < MIGRATION_AUTOMATIC_FACTOR) {
    urgency = 'required';
    deadline = Date.now() + (60 * 60 * 1000); // 1 hour
    message = 'URGENT: Balance exceeds safe threshold for this security level. Migrate immediately!';
  } else if (currentSafetyMargin < MIGRATION_SUGGESTED_FACTOR) {
    urgency = 'suggested';
    deadline = Date.now() + (7 * 24 * 60 * 60 * 1000); // 1 week
    message = 'Your balance is approaching the safe threshold. Consider migrating to a higher level.';
  } else if (futureSafetyMargin < MIGRATION_SUGGESTED_FACTOR) {
    urgency = 'future';
    deadline = Date.now() + (365 * 24 * 60 * 60 * 1000); // 1 year
    message = 'Based on quantum computing projections, consider migrating within the next year.';
  }

  // Find recommended level based on balance value
  let recommendedLevel = currentLevel;
  for (const [level, threshold] of Object.entries(VALUE_THRESHOLDS)) {
    if (balanceUsd < threshold) {
      recommendedLevel = Math.max(parseInt(level), currentLevel);
      break;
    }
  }

  // If safety margin is low, recommend at least next level up
  if (urgency !== 'none' && recommendedLevel === currentLevel) {
    recommendedLevel = Math.min(currentLevel + 1, 15);
  }

  return {
    currentLevel,
    recommendedLevel,
    urgency,
    deadline,
    message,
    safetyMargin: currentSafetyMargin,
    futureSafetyMargin,
    crackingCost: currentCrackingCost,
    futureCrackingCost,
    balanceUsd,
    lqcph: LQCPH_CURRENT,
    quantumBits: levelInfo.quantumBits
  };
}

async function migrateFunds(fromLevel, toLevel, amount, password) {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  if (password !== walletState.sessionPassword) {
    throw new Error('Invalid password');
  }

  if (toLevel <= fromLevel) {
    throw new Error('Can only migrate to higher security levels');
  }

  const activeWallet = getActiveWallet();
  if (!activeWallet) {
    throw new Error('No active wallet');
  }

  const fromWallet = activeWallet.childWallets[fromLevel];
  const toWallet = activeWallet.childWallets[toLevel];

  if (!fromWallet || !toWallet) {
    throw new Error('Invalid wallet levels');
  }

  // Build migration transaction
  const tx = {
    from: fromWallet.address,
    to: toWallet.address,
    value: amount,
    security_level: fromLevel,
    gas_limit: 21000,
    gas_price: '1000000',
    migration: true,
    target_level: toLevel
  };

  const result = await rpcCall('tx_send', [tx]);

  return {
    success: true,
    hash: result.hash || bytesToHex(crypto.getRandomValues(new Uint8Array(32))),
    fromLevel,
    toLevel,
    fromAddress: fromWallet.address,
    toAddress: toWallet.address,
    amount
  };
}

async function sendTransaction(tx, password) {
  if (!walletState.isUnlocked) {
    throw new Error('Wallet is locked');
  }

  if (password !== walletState.sessionPassword) {
    throw new Error('Invalid password');
  }

  const activeParentWallet = getActiveWallet();
  if (!activeParentWallet) {
    throw new Error('No active wallet');
  }

  const activeChildWallet = activeParentWallet.childWallets[walletState.activeLevel];
  if (!activeChildWallet) {
    throw new Error('Active security level not found');
  }

  const transaction = {
    from: activeChildWallet.address,
    to: tx.to,
    value: tx.value,
    security_level: walletState.activeLevel,
    gas_limit: tx.gasLimit || 21000,
    gas_price: tx.gasPrice || '1000000'
  };

  const result = await rpcCall('tx_send', [transaction]);

  // RPC returns the hash directly as a string (e.g., "QH_..."), not as {hash: "..."}
  const txHash = typeof result === 'string' ? result : (result?.hash || null);

  if (!txHash) {
    throw new Error('Transaction failed: no hash returned from node');
  }

  return {
    success: true,
    hash: txHash,
    level: walletState.activeLevel
  };
}

// ============ RPC Client ============

async function rpcCall(method, params) {
  const response = await fetch(currentEndpoint, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      jsonrpc: '2.0',
      method,
      params,
      id: Date.now()
    })
  });

  const data = await response.json();
  if (data.error) {
    throw new Error(data.error.message);
  }
  return data.result;
}

// ============ Encryption Functions ============

async function encryptParentWallet(entropy, password) {
  const encoder = new TextEncoder();

  const salt = crypto.getRandomValues(new Uint8Array(32));
  const iv = crypto.getRandomValues(new Uint8Array(12));

  const keyMaterial = await crypto.subtle.importKey(
    'raw',
    encoder.encode(password),
    'PBKDF2',
    false,
    ['deriveBits', 'deriveKey']
  );

  const key = await crypto.subtle.deriveKey(
    { name: 'PBKDF2', salt, iterations: PBKDF2_ITERATIONS, hash: 'SHA-256' },
    keyMaterial,
    { name: 'AES-GCM', length: 256 },
    false,
    ['encrypt']
  );

  const encrypted = await crypto.subtle.encrypt(
    { name: 'AES-GCM', iv },
    key,
    entropy
  );

  return {
    salt: Array.from(salt),
    iv: Array.from(iv),
    data: Array.from(new Uint8Array(encrypted)),
    version: 1,
    createdAt: Date.now()
  };
}

async function decryptParentWallet(encrypted, password) {
  const encoder = new TextEncoder();

  const salt = new Uint8Array(encrypted.salt);
  const iv = new Uint8Array(encrypted.iv);
  const data = new Uint8Array(encrypted.data);

  const keyMaterial = await crypto.subtle.importKey(
    'raw',
    encoder.encode(password),
    'PBKDF2',
    false,
    ['deriveBits', 'deriveKey']
  );

  const key = await crypto.subtle.deriveKey(
    { name: 'PBKDF2', salt, iterations: PBKDF2_ITERATIONS, hash: 'SHA-256' },
    keyMaterial,
    { name: 'AES-GCM', length: 256 },
    false,
    ['decrypt']
  );

  const decrypted = await crypto.subtle.decrypt(
    { name: 'AES-GCM', iv },
    key,
    data
  );

  return new Uint8Array(decrypted);
}

// ============ Utilities ============

const BASE58_ALPHABET = '123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz';

function base58Encode(bytes) {
  let num = BigInt(0);
  for (const byte of bytes) {
    num = num * BigInt(256) + BigInt(byte);
  }

  let result = '';
  while (num > 0) {
    result = BASE58_ALPHABET[Number(num % BigInt(58))] + result;
    num = num / BigInt(58);
  }

  for (const byte of bytes) {
    if (byte === 0) result = '1' + result;
    else break;
  }

  return result || '1';
}

function bytesToHex(bytes) {
  return Array.from(bytes).map(b => b.toString(16).padStart(2, '0')).join('');
}

function hexToBytes(hex) {
  if (hex.length % 2 !== 0) {
    throw new Error('Invalid hex string');
  }
  const bytes = new Uint8Array(hex.length / 2);
  for (let i = 0; i < bytes.length; i++) {
    bytes[i] = parseInt(hex.substr(i * 2, 2), 16);
  }
  return bytes;
}

// ============ Initialization ============
console.log('QuanChain DTQPE Wallet service worker initialized');
console.log(`Parent wallet size: ${PARENT_WALLET_SIZE} bytes (64 KB)`);
console.log(`Security levels: 1-20 (15 implemented, 5 reserved)`);
